'use strict';

let EmployeeRunners = {Running: {}, Stalled: {}, Completed: {}, Stopped: {}};

function Lifecycle() {
    let self = this;

    self.$rootScope = GetRootScope();
    self._minutesPerTick = 1;
    self._interval = null;
    self._previousDay = self.$rootScope.settings.date.getDay();
    self._previousHour = self.$rootScope.settings.date.getHours();
    self._produceAllTimeout = null;
    self.EmployeeStats = [];

    self.StartTime = (state, avoidDigest) => {
        let $rootScope = self.$rootScope;
        if ($rootScope.contractRequest != null || $rootScope.settings.tierReport != null) return; // Disable when contract or tier report is visible
        $rootScope.settings.paused = false;
        $rootScope.settings.state = state;
        clearInterval(self._interval);

        switch (state) {
            case 1:
                self._minutesPerTick = 1;
                break;
            case 2:
                self._minutesPerTick = 2;
                break;
        }

        self._interval = setInterval(self._update, 100);

        if (avoidDigest == null || !avoidDigest) {
            if (!$rootScope.$$phase) {
                $rootScope.$digest();
            }
        }
    };

    self.PauseTime = (avoidDigest, force) => {
        let $rootScope = self.$rootScope;
        if ($rootScope.contractRequest != null || $rootScope.settings.tierReport != null) return; // Disable when contract or tier report is visible
        if (force != null) {
            $rootScope.settings.paused = force;
        } else {
            $rootScope.settings.paused = !$rootScope.settings.paused;
        }

        if ($rootScope.settings.paused) {
            if (self._interval != null) {
                clearInterval(self._interval);
                PauseKeyboard();
                if (avoidDigest == null || !avoidDigest) {
                    $rootScope.$digest();
                }
            }
        } else {
            self.StartTime($rootScope.settings.state, avoidDigest);
        }

        $rootScope.$broadcast(GameEvents.SpeedChange);
    };

    self._update = (minutesPerTick, avoidBroadcast) => {
        let $rootScope = self.$rootScope;

        minutesPerTick = minutesPerTick == null ? self._minutesPerTick : minutesPerTick;

        $rootScope.settings.date = moment($rootScope.settings.date).add(minutesPerTick, 'minute').toDate();

        // Handle new day
        if ($rootScope.settings.date.getDay() != self._previousDay) {
            self._onNewDay();
        }
        self._previousDay = $rootScope.settings.date.getDay();

        // Handle new hour
        if ($rootScope.settings.date.getHours() != self._previousHour) {
            self._onNewHour();
        }
        self._previousHour = $rootScope.settings.date.getHours();

        // Runners
        self._loadEmployeeStats();
        self._produceAll(avoidBroadcast);
        self._setKeyboardSound();
        self._runEvents();
        self._executeProducts(avoidBroadcast);
        self._runRecruitment(minutesPerTick);
        RunAchievementTriggers();
    };

    self._runRecruitment = (minutesPerTick) => {
        let $rootScope = self.$rootScope;

        if ($rootScope.settings.activeRecruitment != null) {
            $rootScope.settings.activeRecruitment.completedMinutes += minutesPerTick;

            let percentageCompleted = $rootScope.settings.activeRecruitment.completedMinutes * 100 / $rootScope.settings.activeRecruitment.totalMinutes;

            let randomPercentage = (percentageCompleted < 10 ? 50 : 200) * (Object.keys(EmployeeLevels).indexOf($rootScope.settings.activeRecruitment.level) + 1);

            if (_.random(0, randomPercentage) == randomPercentage) {
                let wer = Helpers.CalculateWer().total * 1.2;
                let candidate = Generators.GenerateEmployee(chance.gender(), $rootScope.settings.activeRecruitment.employeeTypeName, $rootScope.settings, $rootScope.settings.activeRecruitment.level, wer);
                $rootScope.settings.candidates.push(candidate);
                $rootScope.addNotification(Helpers.GetLocalized('notification_new_candidate', {name: candidate.name}), 2, candidate.id, null, null, { view: 'employees', tab: 'candidates' });
            }

            // End recruitment
            if ($rootScope.settings.activeRecruitment.completedMinutes >= $rootScope.settings.activeRecruitment.totalMinutes) {
                $rootScope.settings.activeRecruitment = null;
            }
        }
    };

    self._executeProducts = (avoidBroadcast) => {
        let $rootScope = self.$rootScope;

        $rootScope.settings.products.forEach(product => {

            if (product.completed) {
                product.userSegments.forEach(userSegment => {
                    // Increase users
                    let gainedUsers = Helpers.RandomizeNumber(userSegment.usersLeft * userSegment.gainPercentagePerMinute, 10);
                    let gainedUsersByHype = product.hype * gainedUsers / 100;
                    product.users[userSegment.platformName] += gainedUsersByHype;
                    userSegment.usersLeft -= gainedUsersByHype;

                    // Decrease users
                    if (product.hype < 25 && _.sum(product.features.map(x => x.level)) > 3) {
                        let usersToLose = product.users[userSegment.platformName] * 0.000011;
                        usersToLose = usersToLose * (100 + (100 - product.hype)) / 100;

                        product.users[userSegment.platformName] = Math.max(product.users[userSegment.platformName] - usersToLose, 0);
                    }
                });

                // Decrease hype
                let decreaseAmount = _.random(0.00150, 0.00180);
                if (product.hype > 50) {
                    decreaseAmount = decreaseAmount * 3.4;
                }

                product.hype = Math.max(product.hype - decreaseAmount, 0);

                // Run marketing packages
                product.activeMarketingPackages.forEach(marketingPackageResult => {
                    marketingPackageResult.completedMinutes++;
                    if (marketingPackageResult.completedMinutes >= marketingPackageResult.totalMinutes) {
                        product.hype = Math.max(product.hype - marketingPackageResult.hype, 0);
                        _.remove(product.activeMarketingPackages, x => x.marketingPackageName == marketingPackageResult.marketingPackageName);
                    }
                });
            }
        });

        if (!avoidBroadcast) {
            $rootScope.$broadcast(GameEvents.ProductChange);
        }
    };

    self._onNewDay = () => {
        self._runExpenses();
        let $rootScope = self.$rootScope;

        let wer = Helpers.CalculateWer();
        let allWorkstations = $rootScope.settings.office.workstations;
        allWorkstations.forEach(ws => {
            if (ws.employee != null) {
                // Reset overtime and quit if condition met
                if (ws.employee.overtimeMinutes != 0) {
                    ws.employee.overtimeMinutes = 0;
                }

                ws.employee.workingHours = ws.employee.newWorkingHours;

                let employeeStat = Game.Lifecycle.EmployeeStats.find(x => x.employee.id == ws.employee.id);

                // Call In Sick
                if (employeeStat.employeeState != Enums.EmployeeStates.OnVacation) {
                    if (ws.employee.callInSickDaysLeft <= 0) { // Call in sick
                        ws.employee.callInSickDaysLeft = Helpers.RandomizeNumber(Configuration.InitialCallInSickDaysLeft, 40);
                        ws.employee.sickHoursLeft = 24;
                        $rootScope.addNotification(Helpers.GetLocalized('notification_called_in_sick', {name: ws.employee.name}), 4, null, Enums.NotificationTypes.Warning);
                    } else {
                        let base = 250 - wer.total;
                        ws.employee.callInSickDaysLeft -= Math.max(base / 100, 0);
                    }
                }

                // Employee Quits
                if (ws.employee.mood < 25) {
                    EmailGenerator.EmployeeQuit(self.$rootScope.settings.companyName, ws.employee);
                    if (ws.employee.task != null && ws.employee.task.contractId != null) {
                        let contract = $rootScope.settings.contracts.find(x => x.id == ws.employee.task.contractId);
                        if (contract != null) {
                            Helpers.CancelContract(contract);
                        }
                    }
                    RemoveEmployee(ws.employee, true);
                    $rootScope.$broadcast(GameEvents.EmployeeChange);
                }
            }
        });

        // Payout all products
        self._executeProductsDaily();

        Modding._executeEvent('onNewDay', $rootScope.settings);
    };

    self._onNewHour = () => {
        let $rootScope = self.$rootScope;
        self._simulateMoods();
        self._manageContracts();

        // Manage sickhours
        Helpers.GetAllEmployees().filter(x => x.sickHoursLeft > 0).forEach(employee => {
            employee.sickHoursLeft--;
        });

        // Manage time of day
        let hour = $rootScope.settings.date.getHours();

        let timeOfDay = "";
        if (_(hour).inRange(22, 24) || _(hour).inRange(0, 5)) {
            timeOfDay = "night";
        }

        if (_(hour).inRange(5, 7)) {
            timeOfDay = "morning";
        }

        if (_(hour).inRange(18, 22)) {
            timeOfDay = "evening";
        }

        // Update notification counters
        if (Game.Lifecycle.EmployeeStats.some(x => x.isWorking)) { // anyone working
            $rootScope.settings.notifications.filter(x => !x.closed && x.hoursUntilClose > 0).forEach(notification => {
                notification.hoursUntilClose--;
                if (notification.hoursUntilClose <= 0) {
                    notification.closed = true;
                }
            });
        }

        // Update candidates
        $rootScope.settings.candidates.forEach(candidate => {
            candidate.hoursLeft--;
            if (candidate.hoursLeft <= 0) {
                _.remove($rootScope.settings.candidates, x => x.id == candidate.id);
            }
        });

        $rootScope.settings.timeOfDay = timeOfDay;
        Modding._executeEvent('onNewHour', $rootScope.settings);
        $rootScope.$broadcast(GameEvents.OnNewHour);
    };

    self._executeProductsDaily = () => {
        let $rootScope = self.$rootScope;
        let currentDay = GetDateDiffInDays($rootScope.settings.date);
        $rootScope.settings.products.filter(x => !x.underDevelopment).forEach(product => {
            let previousDayStat = _.last(product.stats);

            if (previousDayStat == null)
                product.stats.push({
                    day: currentDay - 1,
                    income: 0,
                    totalUsers: _.sum(_.toArray(product.users)),
                    gainedUsers: _.sum(_.toArray(product.users))
                });

            if (currentDay == previousDayStat.day) {
                console.log('_executeProductsDaily: Current day already exists in stats. Skipping.')
            } else {
                product.ageInDays++;

                let totalUsers = _.sum(_.toArray(product.users));
                let gainedUsers = totalUsers - previousDayStat.totalUsers;

                let productType = ProductTypes.find(x => x.name == product.productTypeName);
                let income = totalUsers * productType.incomePerUserPerDay;

                product.stats.push({
                    day: currentDay,
                    income: income,
                    totalUsers: totalUsers,
                    gainedUsers: gainedUsers
                });

                $rootScope.addNotification(Helpers.GetLocalized('notification_product_revenue', {
                    name: product.name,
                    revenue: numeral(income).format(Configuration.CURRENCY_FORMAT)
                }), 4);

                $rootScope.settings.balance += income;
                $rootScope.addTransaction(Helpers.GetLocalized('transaction_product_revenue', {name: product.name}), income);
            }
        });
    };

    self._manageContracts = () => {
        let $rootScope = self.$rootScope;

        // Manage existing contracts
        $rootScope.settings.contracts.filter(x => !x.completed && x.status == ContractStatuses.Won).forEach(contract => {
            contract.hoursLeft--;

            // Cancel if overdue
            let pastDueFee = Helpers.CalculatePastDueFee(contract);
            let totalPayout = (contract.price - pastDueFee);
            if (totalPayout < 0) {
                let product = CompetitorProducts.find(x => x.id == contract.productId);
                EmailGenerator.ContractCancel(product, contract);
                Helpers.CancelContract(contract);
            }

            // Send notification
            if(contract.hoursLeft == 1) {
                $rootScope.addNotification(Helpers.GetLocalized('notification_contract_deadline', { number: contract.number}), 4);
            }
        });

        $rootScope.$broadcast(GameEvents.ContractChange);
    };

    self._runExpenses = () => {
        let settings = self.$rootScope.settings;

        // Loans
        settings.loans.filter(x => x.active).forEach(loan => {
            settings.balance -= loan.dailyPayment;
            loan.amountLeft -= loan.dailyPayment;
            loan.daysLeft--;

            self.$rootScope.addTransaction(Helpers.GetLocalized('transaction_loan', {
                provider: loan.provider,
                dailyPayment: -loan.dailyPayment
            }), -loan.dailyPayment);

            if (loan.daysLeft <= 0) {
                loan.active = false;
                self.$rootScope.$broadcast(GameEvents.LoanChange);
                EmailGenerator.LoanPaidOut(loan);
            }
        });

        // Office Rent
        let officeRentPerDay = self.$rootScope.activeBuilding.rent / 30;
        settings.balance -= officeRentPerDay;
        self.$rootScope.addTransaction(Helpers.GetLocalized('transaction_office_rent'), -officeRentPerDay);

        // Benefits
        settings.activatedBenefits.map(benefitId => {
            return Database.benefits.find(x => x.id == benefitId)
        }).forEach(benefit => {
            let benefitExpense = Helpers.CalculateBenefitCost(benefit).daily;
            settings.balance -= benefitExpense;
            self.$rootScope.addTransaction(Helpers.GetLocalized(benefit.name), -benefitExpense);
        });

        // Employees
        let allEmployees = Helpers.GetAllEmployees(true);
        allEmployees.forEach(employee => {
            let salary = employee.salary / Configuration.DAYS_IN_MONTH;
            settings.balance -= salary;
            self.$rootScope.addTransaction(Helpers.GetLocalized('transaction_salary', {name: employee.name}), -salary);
        });

        // Product Hosting
        settings.products.forEach(product => {
            let expenses = Helpers.CalculateHostingExpenses(product);
            settings.balance -= expenses;
            self.$rootScope.addTransaction(Helpers.GetLocalized('transaction_hosting', {name: product.name}), -expenses);
        });

    };

    self._simulateMoods = () => {
        let wer = Helpers.CalculateWer();

        self.EmployeeStats.forEach(employeeStat => {
            if (employeeStat.employeeState == Enums.EmployeeStates.OnVacation) {
                employeeStat.employee.mood = Math.min(employeeStat.employee.mood + 1.3, 100);
                employeeStat.employee.vacationHoursLeft--;
            } else {
                if (employeeStat.isEmployeeAtWork) {
                    let exhaustion = Helpers.CalculateHourlyExhaustion(employeeStat.employee);
                    employeeStat.employee.mood = Math.min(Math.max(employeeStat.employee.mood - exhaustion.total, 0), 100);
                } else { // Resting
                    employeeStat.employee.mood = Math.min(employeeStat.employee.mood + 0.05, 100);
                }
            }
        });
    };

    self._setKeyboardSound = () => {
        let anyoneWorking = Game.Lifecycle.EmployeeStats.some(x => x.isWorking);

        if (self.$rootScope.settings.state != 0 && anyoneWorking) {
            PlayKeyboard(self.$rootScope.settings.state == 2);
        } else {
            PauseKeyboard();
        }
    };

    self._produceAll = (avoidBroadcast) => {
        let $rootScope = self.$rootScope;

        self.EmployeeStats.forEach(employeeStat => {
            let employee = employeeStat.employee;

            let producedMinutesAfterBonus = Helpers.CalculateProducedMinutes(employeeStat);

            if (TASK_BASED_EMPLOYEE_TYPES.includes(employee.employeeTypeName)) {
                let task = employee.task;
                if (task != null) {
                    let runner = EmployeeRunners[task.state][employee.employeeTypeName];

                    if (employeeStat.isWorking) {
                        // Increase completedMinutes + XP
                        if (task.state == Enums.TaskStates.Running) {
                            task.completedMinutes += producedMinutesAfterBonus * self._minutesPerTick;
                            $rootScope.settings.xp += 0.1 * self._minutesPerTick;

                            if (task.completedMinutes >= task.totalMinutes) {
                                task.state = Enums.TaskStates.Completed;
                                task.completedMinutes = 0;
                            }
                        }

                        // Execute EmployeeType Runner
                        if (runner) runner($rootScope, employee, task);
                    } else {
                        // Run if task is not running, but the employee is also not work.
                        if (runner && task.state != Enums.TaskStates.Running) runner($rootScope, employee, task);
                    }
                }
            }
        });
        if (!$rootScope.$$phase && !avoidBroadcast) {
            $rootScope.$apply($rootScope.$broadcast(GameEvents.WorkstationChange));
        }
    };

    self._runEvents = () => {
        Database.events.forEach(event => {
            // Check if completable or already completed
            if (!event.completable || !self.$rootScope.settings.completedEvents.some(eventId => eventId == event.id)) {
                let triggered = event.trigger(self.$rootScope);
                if (triggered == true) {
                    if (event.completable) {
                        self.$rootScope.settings.completedEvents.push(event.id);
                    }
                }
            }
        });
    };

    self._loadEmployeeStats = () => {
        let $rootScope = self.$rootScope;
        let now = moment($rootScope.settings.date);

        let getEmployeeState = employee => {
            if (employee == null) return null;

            // Vacation Check
            if (employee.vacationHoursLeft > 0) {
                return Enums.EmployeeStates.OnVacation;
            }

            // Sick Check
            if (employee.sickHoursLeft > 0) {
                return Enums.EmployeeStates.Sick;
            }

            switch (employee.employeeTypeName) {
                default: // Task based employee
                    if (employee.task != null && [Enums.TaskStates.Running, Enums.TaskStates.Stalled].includes(employee.task.state)) {
                        return Enums.EmployeeStates.Working;
                    }
                    break;
                case EmployeeTypeNames.SalesExecutive:
                    if (employee.task != null) {
                        if (employee.task.state == Enums.TaskStates.Stalled && employee.task.contractId == null) {
                            return Enums.EmployeeStates.Awaiting;
                        }
                        if ([Enums.TaskStates.Running, Enums.TaskStates.Stalled].includes(employee.task.state)) {
                            return Enums.EmployeeStates.Working;
                        }
                    }
                    break;
                case EmployeeTypeNames.Manager:
                case EmployeeTypeNames.HrManager:
                    if (employee.employees.length > 0) {
                        return Enums.EmployeeStates.Working;
                    }
                    break;
                case EmployeeTypeNames.LeadDeveloper:
                    if (employee.task != null) {
                        if (employee.task.state == Enums.TaskStates.Running) {
                            return Enums.EmployeeStates.Working;
                        } else {
                            return Enums.EmployeeStates.Awaiting;
                        }
                    }
                    break;
            }

            return Enums.EmployeeStates.Idle;
        };
        let isWorking = function (workstation, isEmployeeAtWork) {
            if (!isEmployeeAtWork || $rootScope.settings.paused) return false;
            switch (workstation.employee.employeeTypeName) {
                default:
                    return workstation.employee.task != null
                        && ![Enums.TaskStates.Stopped, Enums.TaskStates.Stalled].includes(workstation.employee.task.state);
                    break;
                case EmployeeTypeNames.Manager:
                case EmployeeTypeNames.HrManager:
                    return workstation.employee.employees.length > 0;
                break;
            }
        };
        let getWorkstationProgress = (workstation, employeeState) => {
            if ([Enums.EmployeeStates.Sick, Enums.EmployeeStates.Idle].includes(employeeState)) return null;

            if (employeeState == Enums.EmployeeStates.OnVacation) {
                return Math.floor((Configuration.VACATION_HOURS - workstation.employee.vacationHoursLeft) * 100 / Configuration.VACATION_HOURS);
            } else {
                if (TASK_BASED_EMPLOYEE_TYPES.includes(workstation.employee.employeeTypeName)) {
                    let task = workstation.employee.task;
                    if (task == null || task.totalMinutes == 0) return 0;
                    return Math.floor(task.completedMinutes * 100 / task.totalMinutes);
                }

                return null;
            }
        };
        let getWorkstationLabel = (employee, employeeState) => {
            if (employee == null) return null;

            switch (employeeState) {
                case Enums.EmployeeStates.Awaiting:
                    let result = employee.employeeTypeName == EmployeeTypeNames.LeadDeveloper && employee.task != null
                        ? Helpers.GetLocalized('missing_components')
                        : Helpers.GetLocalized('awaiting_instructions');
                    return `<span class="color-blue">${result}</span>`;
                    break;
                case Enums.EmployeeStates.Idle:
                    return Helpers.GetLocalized('idling');
                    break;
                case Enums.EmployeeStates.Sick:
                    return Helpers.GetLocalized('called_in_sick');
                    break;
                case Enums.EmployeeStates.OnVacation:
                    return Helpers.GetLocalized('on_vacation');
                    break;
                case Enums.EmployeeStates.Working:
                    switch (employee.employeeTypeName) {
                        case EmployeeTypeNames.LeadDeveloper:
                            if (employee.task != null) {
                                if (employee.task.awaitingComponent) {
                                    return Helpers.GetLocalized('missing_components');
                                } else {
                                    return Helpers.GetLocalized(employee.task.module.name);
                                }
                            }
                            break;
                        case EmployeeTypeNames.Researcher:
                            if (employee.task != null) {
                                return Helpers.GetLocalized('researching_feature', {feature: Helpers.GetLocalized(employee.task.featureName || employee.task.frameworkName)});
                            }
                            break;
                        case EmployeeTypeNames.Designer:
                        case EmployeeTypeNames.Developer:
                        case EmployeeTypeNames.Marketer:
                        case EmployeeTypeNames.DevOps:
                            if (employee.task != null) {
                                return Helpers.GetLocalized(employee.task.component.name);
                            }
                            break;
                        case EmployeeTypeNames.SalesExecutive:
                            if (employee.task.state == Enums.TaskStates.Stalled) {
                                return Helpers.GetLocalized('managing_contract');
                            } else {
                                return Helpers.GetLocalized('searching_new_contract');
                            }
                            break;
                        case EmployeeTypeNames.Manager:
                        case EmployeeTypeNames.HrManager:
                            if (employee.employees.length == 0) {
                                return '';
                            } else {
                                return Helpers.GetLocalized('managing_employees', {amount: employee.employees.length});
                            }
                            break;
                        default:
                            return 'NOT_DEFINED';
                    }
                    break;
            }
        };
        let shouldMenuFlash = employeeState => {
            return [Enums.EmployeeStates.Idle, Enums.EmployeeStates.Awaiting].includes(employeeState);
        };

        let isEmployeeAtWork = (date, employee, employeeState) => {
            if ([
                    Enums.EmployeeStates.OnVacation,
                    Enums.EmployeeStates.SentHome,
                    Enums.EmployeeStates.Sick
                ].includes(employeeState)) return false;

            let startDate = moment(date).set('hours', employee.workingHours.start.hour).set('minutes', employee.workingHours.start.minute);
            let endDate = moment(date).set('hours', employee.workingHours.end.hour).set('minutes', employee.workingHours.end.minute);
            endDate.add(employee.overtimeMinutes, 'minutes');

            return now.diff(startDate, 'minutes') > 0 && now.diff(endDate, 'minutes') < 0;
        };

        self.EmployeeStats = $rootScope.settings.office.workstations.filter(x => x.employee != null).map(workstation => {
            let employeeState = getEmployeeState(workstation.employee);
            let employeeAtWork = isEmployeeAtWork($rootScope.settings.date, workstation.employee, employeeState);
            return {
                employee: workstation.employee,
                isWorking: isWorking(workstation, employeeAtWork),
                isEmployeeAtWork: employeeAtWork,
                progress: getWorkstationProgress(workstation, employeeState),
                label: getWorkstationLabel(workstation.employee, employeeState),
                shouldMenuFlash: shouldMenuFlash(employeeState),
                employeeState: employeeState,
                cssClass: Helpers.GetEmployeeTypeCssClass(workstation.employee.employeeTypeName),
                productivity: ApplyBonuses(workstation, workstation.employee.speed, $rootScope.officeBonus)
            };
        });

        $rootScope.enableNextDayButton = !self.EmployeeStats.some(x => x.isEmployeeAtWork)
            && self.EmployeeStats.length > 0;
    };
    self._loadEmployeeStats();
}